/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | Website:  https://openfoam.org
    \\  /    A nd           | Copyright (C) 2011-2018 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Description

\*---------------------------------------------------------------------------*/

#include "argList.H"
#include "IOmanip.H"
#include "ODESystem.H"
#include "ODESolver.H"

using namespace Foam;

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //


    float Cpmax = 51540 ;
    float Cnmax = 30550 ;

    float Dp = 3.7e-14;
    float Rp = 40e-6;
    float Lp = 80e-6;
    float kp = 2.33e-11;

    float Ce = 1200;
    float F = 96500;
    float R = 8.314;
    float Ic = -33;
    float Tx = 273+25;

    float Dn = 2e-14;
    float Rn = 40e-6;
    float Ln = 88e-6;
    float kn = 5e-11;
    float solidfracn = 0.52;
    float solidfracp = 0.62;

    float ap = (3*solidfracp)/Rp ;
    float an = (3*solidfracn)/Rn ;

    float delrp = Rp/10;
    float delrn = Rn/10;


    float fp = -(Ic*delrp)/(Dp*F*ap*Lp);
    float fn = -(Ic*delrn)/(Dn*F*an*Ln);

    float k = Dp/pow(delrp,2);
    float k2 = Dn/pow(delrn,2);


    class testODE:

        public ODESystem
        {


        public:

        testODE()
        {}

        label nEqns() const
        {
            return 18;
        }


        void derivatives
        (
            const scalar x,
            const scalarField& y,
            scalarField& dydx
        ) const
        {

            dydx[0] = 2*k*(y[1]-y[0] ) ;

            for(int i = 1 ; i < 8 ; i++) dydx[i] = k*( (1/(i+1))*( y[i+1] - y[i-1] ) + y[i+1] - 2*y[i] + y[i-1] );

            dydx[8] = k*( (1/9)*( +fp/1.5 + (4/3)*y[8] - (4/3)*y[7] ) + ( fp/1.5 - (2/3)*y[8] +(2/3)*y[7] ) );



            dydx[9]  = 2*k2*(  y[10] - y[9] );

            for(int i = 10 ; i < 17 ; i++) dydx[i] = k2*( (1/(i-8))*( y[i+1] - y[i-1] ) + y[i+1] - 2*y[i] + y[i-1] );

            dydx[17] = k2*( (1/9)*( -fn/1.5 + (4/3)*y[17] - (4/3)*y[16] ) + ( -fn/1.5 - (2/3)*y[17] +(2/3)*y[16] ) );


        }

        void jacobian
        (
            const scalar x,
            const scalarField& y,
            scalarField& dfdx,
            scalarSquareMatrix& dfdy
        ) const
        {

            for (int j=0; j<17; j++) dfdx[j] = 0.0;

            for (int j=0; j<18; j++)
            {
              for(k=0; k<18; k++) dfdy(j, k) = 0.0;
            }


            dfdy(0, 0) = -2*k;
            dfdy(0, 1) = 2*k;

            dfdy(1, 0) = 0.5*k;
            dfdy(1, 1) = -2*k;
            dfdy(1, 2) = 1.5*k;

            dfdy(2, 1) = (2/3)*k;
            dfdy(2, 2) = -2*k;
            dfdy(2, 3) = (4/3)*k;

            dfdy(3, 2) = 0.75*k;
            dfdy(3, 3) = -2*k;
            dfdy(3, 4) = 1.25*k;

            dfdy(4, 3) = 0.8*k;
            dfdy(4, 4) = -2*k;
            dfdy(4, 5) = 1.2*k;

            dfdy(5, 4) = (5/6)*k;
            dfdy(5, 5) = -2*k;
            dfdy(5, 6) = (7/6)*k;

            dfdy(6, 5) = (6/7)*k;
            dfdy(6, 6) = -2*k;
            dfdy(6, 7) = (8/7)*k;

            dfdy(7, 6) = 0.875*k;
            dfdy(7, 7) = -2*k;
            dfdy(7, 8) = 1.125*k;

            dfdy(8, 7) = (14/27)*k;
            dfdy(8, 8) = -(14/27)*k;



            dfdy(9, 9) = -2*k2;
            dfdy(9, 10) = 2*k2;

            dfdy(10, 9) = 0.5*k2;
            dfdy(10, 10) = -2*k2;
            dfdy(10, 11) = 1.5*k2;

            dfdy(11, 10) = (2/3)*k2;
            dfdy(11, 11) = -(2)*k2;
            dfdy(11, 12) = (4/3)*k2;

            dfdy(12, 11) = 0.75*k2;
            dfdy(12, 12) = -2*k2;
            dfdy(12, 13) = (1.25)*k2;

            dfdy(13, 12) = 0.8*k2;
            dfdy(13, 13) = -2*k2;
            dfdy(13, 14) = 1.2*k2;

            dfdy(14, 13) = (5/6)*k2;
            dfdy(14, 14) = -(2)*k2;
            dfdy(14, 15) = (7/6)*k2;

            dfdy(15, 14) = (6/7)*k2;
            dfdy(15, 15) = (-2)*k2;
            dfdy(15, 16) = (8/7)*k2;

            dfdy(16, 15) = 0.875*k2;
            dfdy(16, 16) = -2*k2;
            dfdy(16, 17) = 1.125*k2;

            dfdy(17, 16) = (14/27)*k2;
            dfdy(17, 17) = -(14/27)*k2;

        }
    };


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //
// Main program:

void BatteryFunction(int t, float xy[])
{

  testODE ode;
  dictionary dict;
  dict.add("solver", "RKF45");
  autoPtr<ODESolver> odeSolver = ODESolver::New(ode, dict);

  scalar xStart = 0.0;
  scalarField yStart(ode.nEqns());
  for (label i=0; i<9; i++)  yStart[i] = Cpmax*0.5;
  for (label i=9; i<18; i++)  yStart[i] = Cnmax*0.95;

  scalarField dyStart(ode.nEqns());
  ode.derivatives(xStart, yStart, dyStart);

  scalarField y(yStart);
  scalar dxEst = 0.5;
  scalar x = xStart;
  scalar xEnd = t ;

  odeSolver->solve(x, xEnd, y, dxEst);
  float xps= float( +fp/1.5 + (4/3)*y[8] -(1/3)*y[7])/Cpmax;
  float xns= float( -fn/1.5 + (4/3)*y[17] -(1/3)*y[16])/Cnmax;
  xy[0]=xps;
  xy[1]=xns;

}

float PotentialFunction(float xy[])
{
  float Un = float(-0.057 + 0.53*exp(-57*(xy[1])) - 0.184*tanh((20*(xy[1])-21))-0.012*tanh(float(7.57*(xy[1])-4.431)) - 0.0304*tanh(float(18.518*(xy[1])-3.24)) - 0.01*tanh(float(0.255*(xy[1])-0.02653)) );

  float Up = float((-4.656 + 88.669*pow((xy[0]),2) - 401.119*pow((xy[0]),4) + 342.909*pow((xy[0]),6) - 462.471*pow((xy[0]),8) + 433.434*pow((xy[0]),10) )/( -1 + 18.933*pow((xy[0]),2) -79.532*pow((xy[0]),4) + 37.311*pow((xy[0]),6) -73.083*pow((xy[0]),8) + 95.96*pow((xy[0]),10) ));

  float mp = -Ic/(2*F*ap*Lp*kp*Cpmax*sqrt(Ce*xy[0]*(1-xy[0]) ) );

  float etap = 2*R*(Tx/F)*asinh( mp ) ;

  float mn = -Ic/(2*F*an*Ln*kn*Cnmax*sqrt(Ce*xy[1]*(1-xy[1]) ) );

  float etan = 2*R*(Tx/F)*asinh( mn ) ;

  float V = Up - Un - etap - etan ;

  return V;
}

int main(int argc, char *argv[])
{
    int max_step = 3000;
    float Volt[max_step];
    float xy[2];

    for (label i=0; i<max_step; i++)
    {
        Info << "Time :"<< i << endl;
        BatteryFunction(i, xy);
        Info << "the conc  " << xy[0] << " and " << xy[1]<<endl;
        Volt[i] = PotentialFunction(xy);
        Info << "Volt : " <<Volt[i] <<endl;
    }

for (label i=0; i<max_step; i++)  Info << Volt[i] <<endl;


    return 0;



}


// ************************************************************************* //
